/* This code for the structured, nonsegregated model of the bioreactor is */
/*developed in the Purdue Chemical Engineering Ph.D. theses by Jefferson Clay*/
/*Lievense (May 1984) and Atsushi Aoyama (Dec 1994). */
/* The inputs are dilution rate (1/hr) and glucose feed (g/l). */
/* The outputs are cell mass, glucose (substrate), and ethanol (product) all in (g/l) */

#include <math.h>
#include <stdio.h>
#include "cmex.h"
#define NSTATES 7
#define NOUTPUTS 3
#define NINPUTS 4
#define NEEDINPUTS 0

void init_conditions(x0)
double *x0;
/* Steady-state values with dilution = 0.15(1/hr) and glucose in feed = 9.3(g/l)*/
{           
          x0[0] = 4.57655856398896;  /*  x-amount of cell mass (g/l)  */
          x0[1] = 0.03470869881234;  /*  g-amount of substrate glucose (g/l) */
          x0[2] = 1.4148661132222;   /*  e-amount of product ethanol (g/l)*/
          x0[3] = 1.90562813539419;  /*  ef-amount of fermentative enzyme (g/l)  */
          x0[4] = 2.70806017096726;  /*  er-amount of respiratory enzyme (g/l)*/
/*         x0[5] = 0.26; */          /*  mu-specific growth rate (0.26 (1/hr))  */
         /* x[5] for mu is calculated by a lengthy equation below*/
          x0[5] = 0.1499;            /*vef-specific rate of fermentative enzyme sythesis (1/hr)*/
          x0[6] = 0.1716;            /* ver-specific rate of respiratory enzyme synthisis (1/hr)*/
}

/*Function to return derivatives*/
void derivatives(t,x,u,dx)
double t,*x,*u;  /*Input variables*/
double *dx;      /*Output variable*/
{
	double a1,a2,a3,a4,a5,a6;  /*Stoichiometric coefficients for metabolism*/
/* fermentative--(a1*G + X ===>X + (I+e) + a2*E + CO2) with reaction rates r1 and Ef*/    
/* respiratory--(a3*G + a4*O2 +X ===>X + (I+e) + CO2) with reaction rates r2 and Er*/    
/* respiratory--(a5*E + a6*O2 +X ===>X + (I+e) + CO2) with reaction rates r3 and Er*/    
/* anabolism--((I+e) + X ===> 2X + CO2) with reaction rate of r4*/
/* I is the amount of precursors. e is amount of energy.  X=cellmass, E=ethanol, G=glucose*/
/* CO2 is carbon dioxide and O2 is oxygen*/
	double k1,k2,k3,k5,k6;
/*These are rate constants*/
	double kf,kr,ke,ki,kie; 
/* kf,kr,ke are Michaelis-Menten constants, ki,kie are inhibition constants*/
	double ka,kb,n,kc,kd,m; 
/* ka,kb,n are induction constants; kc,kd,m are repression constants */
	double kx,ko,kix,kif,kir; 
/* These are constants associated with the specific growth rates of X,Ef,Er */
	double r1,r2,r3,r4,gf,d,mudist,gdist; 
/* These are reaction rates (r1,r2,r3,r4); gf,d,mudist,gdist are inputs */
	double qg,mu,vef,ver;
/* These are the growth rates for X (mu), Ef, and Er. */
	double mut,veft,vert,qf,qr;
/* mut is the target specific growth rate; veft,vert are the same for Ef,Er; qf is the induction ratio, qf is the repression ratio*/
	double taux,tauf,taur; 
/* These are the time constants for X,Ef,Er*/
	double qo2,qco2,yx,ye;
/* These are the O2 consumption rate (qo2), CO2 production rate (qco2) and the yields for X,E*/
	double vx0,vx1,vx2,vx3,vx4,vx5,vx6,vx7; 
	double vx0i,vx1i,vx2i,vx3i,vx4i,vx5i,vx6i,vx7i;

        d = u[0];        
/* This is the input for dilution rate (1/hr) */
      	gf = u[1];	 
/* This is the input for the glucose in the feed concentration (g/l) */
   	mudist = u[2]-0.26;  
/*This is is the disturbance in the specific growth rate */
    	gdist = u[3];	     
/* This is the disturbance with any glucose that is in the bioreactor before adding glucose, or any miscalculation in the amount present */
/* These are the parameter values for the constants above */
       	a1 = 7.3;
       	a2 = 3.35;
       	a3 = 1.84;
       	a4 = 0.85;
       	a5 = 1.60;
       	a6 = 1.58;
       	k1 = 0.43;
       	k2 = 0.35;
       	k3 = 0.26;
       	k5 = 0.10;
       	k6 = 0.03;
       	kf = 0.20;
       	kr = 0.005;
       	ke = 0.005;
       	ki = 0.10;
      	kie = 0.0005;
       	ka = 20.0;
       	kb = 10.0;
        n = 2.0;
       	kc = 0.10;
       	kd = 7.0;
        m = 1.0;
       	kx = 5.1;
       	ko = 0.043;
      	kix = 0.08;
      	kif = 0.02;
      	kir = 0.08;
    	
    	r1 = k1*x[1]*x[3]/(kf+x[1]);
       	r2 = k2*x[1]*x[4]/(kr+x[1]);
       	qg = (a1*r1+a3*r2)/x[0];
       	r3 = k3*x[2]*x[4]/((ke+qg*(ke/kie)+x[2])*(1+qg/ki));
/* These 4 above are reaction rate calculations for the metabolism reactions above */
        mut = (r1+r2+r3)/x[0];
/* mut is the target specific growth rate--the maximum (1/hr)*/
       	qf = (1+ka*pow(qg,n))/(1+ka*pow(qg,n)+kb);
/* qf is the fermentative enzyme pool induction ratio */
       	qr = (1+kc*pow(qg,m))/(1+kc*pow(qg,m)*(1+kd));
/* qr is the respiratory enzyme pool repression ratio */
     	veft = qf*mut;
/* veft is the target specific growth rate of Ef (1/hr)*/
     	vert = qr*mut;
/* vert is the target specific growth rate of Er (1/hr)*/
       	mu = (((k1*x[1])/(kf+x[1]))*qf)+(qr*((k2*x[1])/(kr+x[1])+(((k3*x[2])/(ke*(1+qg/kie+x[2])))*(1/(1+qg/ki)))))+mudist;
/* mu is the specific growth rate of the cell mass (1/hr) */
     	taux = kx*kix/(mu+kx*ko);
     	tauf = kx*kif/(mu+kx*ko);
     	taur = kx*kir/(mu+kx*ko);
/* These are the residence times of X,Ef,Er*/
       	r4 = mu*x[0];
/* r4 is the reaction rate of anabolism--the synthesis of cell mass */
      	qo2 = 1000.0*(a4*r2+a6*r3)/(32.0*x[0]);
/* qo2 is the rate of oxygen consumption */
     	qco2 = 1000.0*(0.4*(a1*r1+a3*r2)-0.5217*(a2*r1-a5*r3)-0.4435*r4)/(12.0*x[0]);
/* qco2 is the rate of carbon dioxide production */
       	yx = r4/(a1*r1+a3*r2);
/* yx is the yield of cell mass with respect to glucose */
       	ye = (a2*r1-a5*r3)/(a1*r1+a3*r2);
/* ye is the yield of ethanol with respect to glucose */
      
/* The following subroutines result in values for the x[0] through x[6] to stay within model constraints*/
  	vx0i = (d*x[0]*(0.0-1.0)+r4);
    	  if ((x[0]+vx0i) < 0.0)
            vx0 = x[0]*(0.0-1.0);
          else
            vx0 = vx0i;
    
    
  	vx1i = (d*(gf-x[1]+gdist)-(a1*r1+a3*r2));
          if ((x[1]+vx1i) < 0.0)
            vx1 = x[1]*(0.0-1.0);
          else
            vx1 = vx1i;
    
  	vx2i = (d*x[2]*(0.0-1.0)+(a2*r1-a5*r3));  
          if ((x[2]+vx2i) < 0.0)
            vx2 = x[2]*(0.0-1.0);
          else
            vx2 = vx2i;
        
  	vx3i = (d*x[3]*(0.0-1.0)+(x[5]*x[0]-k5*x[3]));  
          if ((x[3]+vx3i) < 0.0)
            vx3 = x[3]*(0.0-1.0);
          else
            vx3 = vx3i;
      
  	vx4i = (d*x[4]*(0.0-1.0)+(x[6]*x[0]-k6*x[4]));
          if ((x[4]+vx4i) < 0.0)
            vx4 = x[4]*(0.0-1.0);
          else
            vx4 = vx4i;
      
/*  	vx5i = (mut - x[5] + mudist)/taux; */
/*        if (x[5]  = mut)*/
/*          vx5 = 0.0;*/
/*        else*/
/*           if((x[5]+vx5i) > mut)*/
/*             vx5 = mut-x[5];*/
/*           else*/
/*             vx5 = vx5i;*/
           
    
  	vx6i = (veft-x[5])/tauf;
          if (x[5] = veft)
            vx6 = 0.0;
          else
            if ((x[5]+vx6i) > veft)
              vx6 = veft-x[5];
            else
              vx6 = vx6i;
        
  	vx7i = (vert-x[6])/taur;
          if (x[6] = vert)
            vx7 = 0.0;
          else
            if ((x[6]+vx7i) > vert)
              vx7 = vert-x[6];
            else
              vx7 = vx7i;
     
   	dx[0] = vx0;
   	dx[1] = vx1;
   	dx[2] = vx2;
   	dx[3] = vx3;
   	dx[4] = vx4;
   	dx[5] = vx6;
   	dx[6] = vx7;
/*  	 dx[7] = vx7; */ 
}   
    
/*Function to return outputs*/
void outputs(t,x,u,y)
double t,*x,*u;  /*Input variables*/
double *y;        /*Output variable*/
{
	
  	y[0] = x[0]; 
/* y[0] is cell mass (g/l) */
  	y[1] = x[1];
/* y[1] is glucose (g/l) */
  	y[2] = x[2];
/* y[2] is ethanol (g/l) */
}

#include "simulink.h"
