
#define S_FUNCTION_NAME biocore2

/*
 * Need to include simstruc.h for the definition of the SimStruct and
 * its associated macro definitions.
 */

#include "simstruc.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "mex.h"

/*
 * mdlInitializeSizes - initialize the sizes array
 *
 * The sizes array is used by SIMULINK to determine the S-function block's
 * characteristics (number of inputs, outputs, states, etc.).
 *
 * The direct feedthrough flag can be either 1=yes or 0=no. It should be
 * set to 1 if the input, "u", is used is the mdlOutput function. Setting this
 * to 0 is akin to making a promise that "u" will not be used in the mdlOutput
 * function. If you break the promise, then unpredictable results will occur.
 */
static void mdlInitializeSizes(SimStruct *S)
{
    ssSetNumContStates(    S, 7);   /* number of continuous states           */
    ssSetNumDiscStates(    S, 0);   /* number of discrete states             */
    ssSetNumInputs(        S, 4);   /* number of inputs                      */
    ssSetNumOutputs(       S, 3);   /* number of outputs                     */
    ssSetDirectFeedThrough(S, 1);   /* direct feedthrough flag               */
    ssSetNumSampleTimes(   S, 1);   /* number of sample times                */
    ssSetNumSFcnParams(    S, 0);   /* number of input arguments             */
    ssSetNumRWork(         S, 0);   /* number of real work vector elements   */
    ssSetNumIWork(         S, 0);   /* number of integer work vector elements*/
    ssSetNumPWork(         S, 0);   /* number of pointer work vector elements*/
}

/*
 * mdlInitializeSampleTimes - initialize the sample times array
 *
 * This function is used to specify the sample time(s) for your S-function.
 * You must register the same number of sample times as specified in 
 * ssSetNumSampleTimes. If you specify that you have no sample times, then
 * the S-function is assumed to have one inherited sample time.
 *
 * The sample times are specified period, offset pairs. The valid pairs are:
 *
 *   [CONTINUOUS_SAMPLE_TIME   0     ]  : Continuous sample time.
 *   [period                   offset]  : Discrete sample time where
 *					  period > 0.0 and offset < period or
 *				          equal to 0.0.
 *   [VARIABLE_SAMPLE_TIME     0     ]  : Variable step discrete sample time
 *				          where mdlGetTimeOfNextVarHit is
 *				          called to get the time of the next
 *					  sample hit.
 *
 *  or you can specify that the sample time is inherited from the driving
 *  block in which case the S-function can have only one sample time:
 *    [INHERITED_SAMPLE_TIME    0    ]
 */
static void mdlInitializeSampleTimes(SimStruct *S)
{
    ssSetSampleTime(S, 0, CONTINUOUS_SAMPLE_TIME);
    ssSetOffsetTime(S, 0, 0.0);
    
    /*
     * SET OTHER SAMPLE TIMES AND OFFSETS HERE
     */
}


/*
 * mdlInitializeConditions - initialize the states
 *
 * In this function, you should initialize the continuous and discrete
 * states for your S-function block.  The initial states are placed
 * in the x0 variable.  You can also perform any other initialization
 * activities that your S-function may require.
 */
static void mdlInitializeConditions(double *x0, SimStruct *S)
{
	x0[0] = 4.57655856398896;  /*  x-amount of cell mass (g/l)  */
	x0[1] = 0.03470869881234;  /*  g-amount of substrate glucose (g/l) */
	x0[2] = 1.4148661132222;   /*  e-amount of product ethanol (g/l)*/
	x0[3] = 1.90562813539419;  /*  ef-amount of fermentative enzyme (g/l)  */
	x0[4] = 2.70806017096726;  /*  er-amount of respiratory enzyme (g/l)*/
	/*x0[5] = 0.26; */ /*  mu-specific growth rate (0.26 (1/hr))  */
	/* x[5] for mu is calculated by a lengthy equation below*/
	x0[5] = 0.1499;/*vef-specific rate of fermentative enzyme sythesis (1/hr)*/
	x0[6] = 0.1716;/* ver-specific rate of respiratory enzyme synthisis (1/hr)*/
}

/*
 * mdlOutputs - compute the outputs
 *
 * In this function, you compute the outputs of your S-function
 * block.  The outputs are placed in the y variable.
 */

static void mdlOutputs(double *y, double *x, double *u, SimStruct *S, int tid)
{
	y[0] = x[0]; /* cell mass (g/l) */
  	y[1] = x[1]; /* glucose (g/l) */
  	y[2] = x[2]; /* ethanol (g/l) */
}

/*
 * mdlUpdate - perform action at major integration time step
 *
 * This function is called once for every major integration time step.
 * Discrete states are typically updated here, but this function is useful
 * for performing any tasks that should only take place once per integration
 * step.
 */

static void mdlUpdate(double *x, double *u, SimStruct *S, int tid)
{
    /*
     * YOUR CODE GOES HERE
     */
}

/*
 * mdlDerivatives - compute the derivatives
 *
 * In this function, you compute the S-function block's derivatives.
 * The derivatives are placed in the dx variable.
 */

static void mdlDerivatives(double *dx, double *x, double *u, SimStruct *S, int tid)
{
	double a1,a2,a3,a4,a5,a6;  /*Stoichiometric coefficients for metabolism*/
	/* fermentative--(a1*G + X ===>X + (I+e) + a2*E + CO2) with reaction rates r1 and Ef*/
	/* respiratory--(a3*G + a4*O2 +X ===>X + (I+e) + CO2) with reaction rates r2 and Er*/
	/* respiratory--(a5*E + a6*O2 +X ===>X + (I+e) + CO2) with reaction rates r3 and Er*/
	/* anabolism--((I+e) + X ===> 2X + CO2) with reaction rate of r4*/
	/* I is the amount of precursors. e is amount of energy.  X=cellmass, E=ethanol, G=glucose*/
	/* CO2 is carbon dioxide and O2 is oxygen*/
	double k1,k2,k3,k5,k6;
		/*These are rate constants*/
	double kf,kr,ke,ki,kie; 
		/* kf,kr,ke are Michaelis-Menten constants, ki,kie are inhibition constants*/
	double ka,kb,n,kc,kd,m; 
		/* ka,kb,n are induction constants; kc,kd,m are repression constants */
	double kx,ko,kix,kif,kir; 
		/* These are constants associated with the specific growth rates of X,Ef,Er */
	double r1,r2,r3,r4,gf,d,mudist;
		/* These are reaction rates (r1,r2,r3,r4); gf,d,mudist are inputs */
	double qg,mu,vef,ver;
		/* These are the growth rates for X (mu), Ef, and Er. */
	double mut,veft,vert,qf,qr;
		/* mut is the target specific growth rate; veft,vert are the same for Ef,Er; qf is the induction ratio, qf is the repression ratio*/
	double taux,tauf,taur; 
		/* These are the time constants for X,Ef,Er*/
	double qo2,qco2,yx,ye;
		/* These are the O2 consumption rate (qo2), CO2 production rate (qco2) and the yields for X,E*/
	double vx0,vx1,vx2,vx3,vx4,vx5,vx6,vx7; 
	double vx0i,vx1i,vx2i,vx3i,vx4i,vx5i,vx6i,vx7i;

   d = u[0];
		/* This is the input for dilution rate (1/hr) */
   gf = u[1];
		/* This is the input for the glucose in the feed concentration (g/l) */
   mudist = u[2]-0.26;
		/*This is is the disturbance in the specific growth rate */
   qg = u[3];
		/* This is the specific glucose uptake rate (disturbance variable) */

   /* These are the parameter values for the constants above */
   	a1 = 7.3;
      a2 = 3.35;
      a3 = 1.84;
      a4 = 0.85;
      a5 = 1.60;
      a6 = 1.58;
      k1 = 0.43;
      k2 = 0.35;
      k3 = 0.26;
      k5 = 0.10;
      k6 = 0.03;
      kf = 0.20;
      kr = 0.005;
      ke = 0.005;
      ki = 0.10;
      kie = 0.0005;
      ka = 20.0;
      kb = 10.0;
      n = 2.0;
      kc = 0.10;
      kd = 7.0;
      m = 1.0;
      kx = 5.1;
      ko = 0.043;
      kix = 0.08;
      kif = 0.02;
      kir = 0.08;
    	
   r1 = k1*x[1]*x[3]/(kf+x[1]);
   r2 = k2*x[1]*x[4]/(kr+x[1]);
   /*qg = (a1*r1+a3*r2)/x[0];  qg is now an input disturbance */
   r3 = k3*x[2]*x[4]/((ke+qg*(ke/kie)+x[2])*(1+qg/ki));
		/* These 4 above are reaction rate calculations for the metabolism reactions above */
   mut = (r1+r2+r3)/x[0];
		/* mut is the target specific growth rate--the maximum (1/hr)*/
   qf = (1+ka*pow(qg,n))/(1+ka*pow(qg,n)+kb);
		/* qf is the fermentative enzyme pool induction ratio */
   qr = (1+kc*pow(qg,m))/(1+kc*pow(qg,m)*(1+kd));
		/* qr is the respiratory enzyme pool repression ratio */
   veft = qf*mut;
		/* veft is the target specific growth rate of Ef (1/hr)*/
   vert = qr*mut;
		/* vert is the target specific growth rate of Er (1/hr)*/
   mu = (((k1*x[1])/(kf+x[1]))*qf)+(qr*((k2*x[1])/(kr+x[1])+(((k3*x[2])/(ke*(1+qg/kie+x[2])))*(1/(1+qg/ki)))))+mudist;
		/* mu is the specific growth rate of the cell mass (1/hr) */
   taux = kx*kix/(mu+kx*ko);
   tauf = kx*kif/(mu+kx*ko);
   taur = kx*kir/(mu+kx*ko);
		/* These are the residence times of X,Ef,Er*/
   r4 = mu*x[0];
		/* r4 is the reaction rate of anabolism--the synthesis of cell mass */
   qo2 = 1000.0*(a4*r2+a6*r3)/(32.0*x[0]);
		/* qo2 is the rate of oxygen consumption */
   qco2 = 1000.0*(0.4*(a1*r1+a3*r2)-0.5217*(a2*r1-a5*r3)-0.4435*r4)/(12.0*x[0]);
		/* qco2 is the rate of carbon dioxide production */
   yx = r4/(a1*r1+a3*r2);
		/* yx is the yield of cell mass with respect to glucose */
   ye = (a2*r1-a5*r3)/(a1*r1+a3*r2);
		/* ye is the yield of ethanol with respect to glucose */
      
/* The following subroutines result in values for the x[0] through x[6] to stay within model constraints*/
  	vx0i = (d*x[0]*(0.0-1.0)+r4);
    	  if ((x[0]+vx0i) < 0.0)
            vx0 = x[0]*(0.0-1.0);
          else
            vx0 = vx0i;

    
  	vx1i = (d*(gf-x[1])-(a1*r1+a3*r2));
          if ((x[1]+vx1i) < 0.0)
            vx1 = x[1]*(0.0-1.0);
          else
            vx1 = vx1i;
    
  	vx2i = (d*x[2]*(0.0-1.0)+(a2*r1-a5*r3));  
          if ((x[2]+vx2i) < 0.0)
            vx2 = x[2]*(0.0-1.0);
          else
            vx2 = vx2i;
        
  	vx3i = (d*x[3]*(0.0-1.0)+(x[5]*x[0]-k5*x[3]));  
          if ((x[3]+vx3i) < 0.0)
            vx3 = x[3]*(0.0-1.0);
          else
            vx3 = vx3i;
      
  	vx4i = (d*x[4]*(0.0-1.0)+(x[6]*x[0]-k6*x[4]));
          if ((x[4]+vx4i) < 0.0)
            vx4 = x[4]*(0.0-1.0);
          else
            vx4 = vx4i;
      
/*  	vx5i = (mut - x[5] + mudist)/taux; */
/*        if (x[5]  = mut)*/
/*          vx5 = 0.0;*/
/*        else*/
/*           if((x[5]+vx5i) > mut)*/
/*             vx5 = mut-x[5];*/
/*           else*/
/*             vx5 = vx5i;*/
           
    
  	vx6i = (veft-x[5])/tauf;
          if (x[5] = veft)
            vx6 = 0.0;
          else
            if ((x[5]+vx6i) > veft)
              vx6 = veft-x[5];
            else
              vx6 = vx6i;
        
  	vx7i = (vert-x[6])/taur;
          if (x[6] = vert)
            vx7 = 0.0;
          else
            if ((x[6]+vx7i) > vert)
              vx7 = vert-x[6];
            else
              vx7 = vx7i;
     
   	dx[0] = vx0;
   	dx[1] = vx1;
   	dx[2] = vx2;
   	dx[3] = vx3;
   	dx[4] = vx4;
   	dx[5] = vx6;
   	dx[6] = vx7;
		/* dx[7] = vx7; */
}


#if 0  /* Change to a 1 if VARIABLE_SAMPLE_TIME has been specified */
/*
 * mdlGetTimeOfNextVarHit - Get the time of the next variable sample time hit
 *
 * This function is called one for every major integration time step.
 * It must return time of next hit by using ssSetTNext. The time of the
 * next hit must be greater than ssGetT(S).
 */
#define MDL_GET_TIME_OF_NEXT_VAR_HIT
static void mdlGetTimeOfNextVarHit(SimStruct *S)
{
    ssSetTNext(S, <timeOfNextHit>);
}
#endif

/*
 * mdlTerminate - called when the simulation is terminated.
 *
 * In this function, you should perform any actions that are necessary
 * at the termination of a simulation.  For example, if memory was allocated
 * in mdlInitializeConditions, this is the place to free it.
 */

static void mdlTerminate(SimStruct *S)
{
    /*
     * YOUR CODE GOES HERE
     */
}

#ifdef	MATLAB_MEX_FILE    /* Is this file being compiled as a MEX-file? */
#include "simulink.c"      /* MEX-file interface mechanism */
#else
#include "cg_sfun.h"       /* Code generation registration function */
#endif
