
#define S_FUNCTION_NAME bionew4

/*
 * Need to include simstruc.h for the definition of the SimStruct and
 * its associated macro definitions.
 */

#include "simstruc.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "mex.h"

/*
 * mdlInitializeSizes - initialize the sizes array
 *
 * The sizes array is used by SIMULINK to determine the S-function block's
 * characteristics (number of inputs, outputs, states, etc.).
 *
 * The direct feedthrough flag can be either 1=yes or 0=no. It should be
 * set to 1 if the input, "u", is used is the mdlOutput function. Setting this
 * to 0 is akin to making a promise that "u" will not be used in the mdlOutput
 * function. If you break the promise, then unpredictable results will occur.
 */
static void mdlInitializeSizes(SimStruct *S)
{
    ssSetNumContStates(    S, 11);   /* number of continuous states           */
    ssSetNumDiscStates(    S, 0);   /* number of discrete states             */
    ssSetNumInputs(        S, 5);   /* number of inputs                      */
    ssSetNumOutputs(       S, 4);   /* number of outputs                     */
    ssSetDirectFeedThrough(S, 0);   /* direct feedthrough flag               */
    ssSetNumSampleTimes(   S, 1);   /* number of sample times                */
    ssSetNumSFcnParams(    S, 0);   /* number of input arguments             */
    ssSetNumRWork(         S, 0);   /* number of real work vector elements   */
    ssSetNumIWork(         S, 0);   /* number of integer work vector elements*/
    ssSetNumPWork(         S, 0);   /* number of pointer work vector elements*/
}

/*
 * mdlInitializeSampleTimes - initialize the sample times array
 *
 * This function is used to specify the sample time(s) for your S-function.
 * You must register the same number of sample times as specified in 
 * ssSetNumSampleTimes. If you specify that you have no sample times, then
 * the S-function is assumed to have one inherited sample time.
 *
 * The sample times are specified period, offset pairs. The valid pairs are:
 *
 *   [CONTINUOUS_SAMPLE_TIME   0     ]  : Continuous sample time.
 *   [period                   offset]  : Discrete sample time where
 *					  period > 0.0 and offset < period or
 *				          equal to 0.0.
 *   [VARIABLE_SAMPLE_TIME     0     ]  : Variable step discrete sample time
 *				          where mdlGetTimeOfNextVarHit is
 *				          called to get the time of the next
 *					  sample hit.
 *
 *  or you can specify that the sample time is inherited from the driving
 *  block in which case the S-function can have only one sample time:
 *    [INHERITED_SAMPLE_TIME    0    ]
 */
static void mdlInitializeSampleTimes(SimStruct *S)
{
    ssSetSampleTime(S, 0, CONTINUOUS_SAMPLE_TIME);
    ssSetOffsetTime(S, 0, 0.0);
    
    /*
     * SET OTHER SAMPLE TIMES AND OFFSETS HERE
     */
}


/*
 * mdlInitializeConditions - initialize the states
 *
 * In this function, you should initialize the continuous and discrete
 * states for your S-function block.  The initial states are placed
 * in the x0 variable.  You can also perform any other initialization
 * activities that your S-function may require.
 */
static void mdlInitializeConditions(double *x0, SimStruct *S)
{
        x0[0]  = 0.005; /* S1[0] = S1f */
        x0[1]  = 0.005; /* S2[0] = S2f */
        x0[2]  = 1.2; /* S3[0] = S3f */
        x0[3]  = 0; /* p1 */
        x0[4]  = 0; /* p2 */
        x0[5]  = 0; /* C */
        x0[6]  = 0.01; /* ea1 */
        x0[7]  = 0.01; /* eb1 */
        x0[8]  = 0.01; /* e2 */
        x0[9]  = 0.01; /* e3 */
        x0[10] = 0.01; /* eg */
}

/*
 * mdlOutputs - compute the outputs
 *
 * In this function, you compute the outputs of your S-function
 * block.  The outputs are placed in the y variable.
 */

static void mdlOutputs(double *y, double *x, double *u, SimStruct *S, int tid)
{
     y[0] = x[0];
     y[1] = x[1];
     y[2] = x[2];
     y[3] = x[5];
}

/*
 * mdlUpdate - perform action at major integration time step
 *
 * This function is called once for every major integration time step.
 * Discrete states are typically updated here, but this function is useful
 * for performing any tasks that should only take place once per integration
 * step.
 */

static void mdlUpdate(double *x, double *u, SimStruct *S, int tid)
{
    /*
     * YOUR CODE GOES HERE
     */
}

/*
 * mdlDerivatives - compute the derivatives
 *
 * In this function, you compute the S-function block's derivatives.
 * The derivatives are placed in the dx variable.
 */

static void mdlDerivatives(double *dx, double *x, double *u, SimStruct *S, int tid)
{
   double ra1, ka1, eamax1, Kaa;
   double rb1, kb1, ebmax1, Kbb;
   double rea1, alphaea1, Kea1;
   double reb1, alphaeb1, Keb1;
   double r2, k2, emax2, K2;
   double re2, alphae2, Ke2;
   double r3, k3, emax3, K3;
   double re3, alpha3, Ke3;
   double rg, umaxg, emaxg, K1g, K2g;
   double reg, alphag, K1eg, K2eg;
   double ua1, ub1;
   double va1, vb1;
   double u2, u3, v2, v3;
   double ug, vg;
   double D, Sf1, Sf2, Sf3, cf, B;

/* Inputs */
   Sf1 = u[0];  /* S1 Feed Composition */
   Sf2 = u[1];  /* S2 Feed Composition */
   Sf3 = u[2];  /* S3 Feed Composition */
   D = u[3];	 /* Dilution Rate */
   cf = u[4];   /* Cell Biomass Feed Conc. */

/* Parameters */

   B 			= 0.05;
   ka1 		= 1.0;	/* adjustable */
   eamax1 	= 0.044;
   Kaa 		= 0.01;
   kb1 		= 0.1;	/* adjustable */
   ebmax1 	= 0.044;
   Kbb 		= 0.01;
   alphaea1 = 0.01;
   Kea1 		= 0.01;
   alphaeb1 = 0.01;
   Keb1 		= 0.01;
   k2 		= 0.1;	/* adjustable */
   emax2 	= 0.044;
   K2 		= 0.01;
   alphae2 	= 0.01;
   Ke2 		= 0.01;
   k3 		= 0.1;	/* adjustable */
   emax3 	= 0.044;
   alpha3 	= 0.01;
   Ke3 		= 0.01;
   umaxg 	= 0.15;
   emaxg 	= 0.2;
   K1g 		= 0.01;
   K2g 		= 0.01;
   alphag 	= 0.01;
   K1eg 		= 0.01;
   K2eg 		= 0.01;

/* Physical Processes */

   ra1 = ka1 * x[6]/eamax1 * x[0]/(Kaa + x[0]);
   rb1 = kb1 * x[7]/ebmax1 * x[0]/(Kbb + x[0]);

   rea1 = alphaea1 * x[0]/(Kea1 + x[0]);
   reb1 = alphaeb1 * x[0]/(Keb1 + x[0]);

   r2 = k2 * x[8]/emax2 * x[1]/(K2 + x[1]);
   re2 = alphae2 * x[1]/(Ke2 + x[1]);
   r3 = k3 * x[9]/emax3 * x[2]/(K3 + x[2]);
   re3 = alpha3 * x[2]/(Ke3 + x[2]);
   rg = umaxg * x[10]/emaxg * x[3]/(K1g + x[3]) * x[4]/(K2g + x[4]);
   reg = alphag * x[3]/(K1eg + x[3]) * x[4]/(K2eg + x[4]);

/* Pathway Regulation */

   ua1 = ra1/(ra1 + r2) * ra1/x[3]/(ra1/x[3] + rb1/x[4]);
   ub1 = rb1/(rb1 + r3) * rb1/x[4]/(rb1/x[4] + ra1/x[3]);
   va1 = (ra1/__max(ra1,r2))*(ra1/x[3]/__max(rb1/x[4],ra1/x[3]));
   vb1 = (rb1/__max(rb1,r3))*(rb1/x[4]/__max(rb1/x[4],ra1/x[3]));
   u2 = r2/(r2 + ra1);
   u3 = r3/(r3 + rb1);
   v2 = r2/__max(r2,ra1);
   v3 = r3/__max(r3,rb1);
   ug = 1;
   vg = 1;

/* Model Formulation */

   dx[0]  = D*(Sf1 - x[0]) - ra1*va1*x[5] - rb1*vb1*x[5];
   dx[1]  = D*(Sf2 - x[1]) - r2*x[5];
   dx[2]  = D*(Sf3 - x[2]) - r3*x[5];
   dx[3]  = ra1*va1 + r2*u2 - rg*vg - rg*x[3];
   dx[4]  = rb1*vb1 + r3*v3 - rg*vg - rg*x[4];
   dx[5]  = D*(cf - x[5]) + rg*x[5];
   dx[6]  = rea1*ua1 - (rg + B)*x[6];
   dx[7]  = reb1*ub1 - (rg + B)*x[7];
   dx[8]  = re2*u2 - (rg + B)*x[8];
   dx[9]  = re3*u3 - (rg + B)*x[9];
   dx[10] = reg*ug - (rg + B)*x[10];
}


#if 0  /* Change to a 1 if VARIABLE_SAMPLE_TIME has been specified */
/*
 * mdlGetTimeOfNextVarHit - Get the time of the next variable sample time hit
 *  
 * This function is called one for every major integration time step.
 * It must return time of next hit by using ssSetTNext. The time of the
 * next hit must be greater than ssGetT(S).
 */
#define MDL_GET_TIME_OF_NEXT_VAR_HIT
static void mdlGetTimeOfNextVarHit(SimStruct *S)
{
    ssSetTNext(S, <timeOfNextHit>);
}
#endif

/*
 * mdlTerminate - called when the simulation is terminated.
 *
 * In this function, you should perform any actions that are necessary
 * at the termination of a simulation.  For example, if memory was allocated
 * in mdlInitializeConditions, this is the place to free it.
 */

static void mdlTerminate(SimStruct *S)
{
    /*
     * YOUR CODE GOES HERE
     */
}

#ifdef	MATLAB_MEX_FILE    /* Is this file being compiled as a MEX-file? */
#include "simulink.c"      /* MEX-file interface mechanism */
#else
#include "cg_sfun.h"       /* Code generation registration function */
#endif
