
#define S_FUNCTION_NAME pufcore4

/*
 * Need to include simstruc.h for the definition of the SimStruct and
 * its associated macro definitions.
 */

#include "simstruc.h"
#include "stdio.h"
#include "math.h"
/* #include "mex.h"*/

/* number of partitions */
#define N 20

/*
 * mdlInitializeSizes - initialize the sizes array
 *
 * The sizes array is used by SIMULINK to determine the S-function block's
 * characteristics (number of inputs, outputs, states, etc.).
 *
 * The direct feedthrough flag can be either 1=yes or 0=no. It should be
 * set to 1 if the input, "u", is used is the mdlOutput function. Setting this
 * to 0 is akin to making a promise that "u" will not be used in the mdlOutput
 * function. If you break the promise, then unpredictable results will occur.
 */
static void mdlInitializeSizes(SimStruct *S)
{
    ssSetNumContStates(    S, 20+6);   /* number of continuous states         */
    ssSetNumDiscStates(    S, 0);   /* number of discrete states             */
    ssSetNumInputs(        S, 7);   /* number of inputs                      */
    ssSetNumOutputs(       S, 4);   /* number of outputs                     */
    ssSetDirectFeedThrough(S, 1);   /* direct feedthrough flag               */
    ssSetNumSampleTimes(   S, 1);   /* number of sample times                */
    ssSetNumSFcnParams(    S, 0);   /* number of input arguments             */
    ssSetNumRWork(         S, 0);   /* number of real work vector elements   */
    ssSetNumIWork(         S, 0);   /* number of integer work vector elements*/
    ssSetNumPWork(         S, 0);   /* number of pointer work vector elements*/
}

/*
 * mdlInitializeSampleTimes - initialize the sample times array
 *
 * This function is used to specify the sample time(s) for your S-function.
 * You must register the same number of sample times as specified in 
 * ssSetNumSampleTimes. If you specify that you have no sample times, then
 * the S-function is assumed to have one inherited sample time.
 *
 * The sample times are specified period, offset pairs. The valid pairs are:
 *
 *   [CONTINUOUS_SAMPLE_TIME   0     ]  : Continuous sample time.
 *   [period                   offset]  : Discrete sample time where
 *					  period > 0.0 and offset < period or
 *				          equal to 0.0.
 *   [VARIABLE_SAMPLE_TIME     0     ]  : Variable step discrete sample time
 *				          where mdlGetTimeOfNextVarHit is
 *				          called to get the time of the next
 *					  sample hit.
 *
 *  or you can specify that the sample time is inherited from the driving
 *  block in which case the S-function can have only one sample time:
 *    [INHERITED_SAMPLE_TIME    0    ]
 */
static void mdlInitializeSampleTimes(SimStruct *S)
{
    ssSetSampleTime(S, 0, CONTINUOUS_SAMPLE_TIME);
    ssSetOffsetTime(S, 0, 0.0);
    
    /*
     * SET OTHER SAMPLE TIMES AND OFFSETS HERE
     */
}


/*
 * mdlInitializeConditions - initialize the states
 *
 * In this function, you should initialize the continuous and discrete
 * states for your S-function block.  The initial states are placed
 * in the x0 variable.  You can also perform any other initialization
 * activities that your S-function may require.
 */
static void mdlInitializeConditions(double *x0, SimStruct *S)
{
        x0[0] = 0.0005368540297717153;	/* Methane */
        x0[1] = 1.662591579255257;	/* Water */
        x0[2] = 0.9220837715711637;	/* Oxygen */
        x0[3] = 0.0009888710608306974;	/* Carbon Monoxide */
        x0[4] = 0.8303069185667976;	/* Carbon Dioxide */


	x0[5] = 3.273256980550678e+02;  /* Partion Temperatures (1 - 20) (K) */
	x0[5+1] = 3.443825309147320e+02;
	x0[5+2] = 3.611746709068880e+02;
	x0[5+3] = 3.777062256120392e+02;
	x0[5+4] = 3.939812388680670e+02;
	x0[5+5] = 4.100036917594059e+02;
	x0[5+6] = 4.257775035908704e+02;
	x0[5+7] = 4.413065328463692e+02;
	x0[5+8] = 4.565945781327409e+02;
	x0[5+9] = 4.716453791089443e+02;
	x0[5+10] = 4.864626174008284e+02;
	x0[5+11] = 5.010499175017071e+02;
	x0[5+12] = 5.154108476589583e+02;
	x0[5+13] = 5.295489207468647e+02;
	x0[5+14] = 5.434675951259094e+02;
	x0[5+15] = 5.571702754887368e+02;
	x0[5+16] = 5.706603136929862e+02;
	x0[5+17] = 5.839410095811995e+02;
	x0[5+18] = 5.970156117880078e+02;
	x0[5+19] = 6.098873185347901e+02;

   /* Furnace Temperature (K) */
        x0[20+5] = 1.426469584859546e+03;
}

/*
 * mdlOutputs - compute the outputs
 *
 * In this function, you compute the outputs of your S-function
 * block.  The outputs are placed in the y variable.
 */

static void mdlOutputs(double *y, double *x, double *u, SimStruct *S, int tid)
{
    	double igd,no2i,nh2oi,exito2; 
	double afr,at,fgfr,fgp,P,R;

     /* Inputs */
	afr=u[2];	/* Air Flow Rate (m3/min) */
	at=u[3];	/* Air Temperature */
	fgfr=u[4];	/* Fuel Gas Flow Rate (m3/min) */
	fgp=u[6];	/* Fuel Gas Purity (mol CH4/mol) */

     /* other constants */
	P=1.0;                  /* pressure */
	R=8.206e-05;           /* ideal gas constant (m^3-atm/mol-K) */

        igd = P / (R * at);		/* Ideal Gas Density at Feed Temperature */
        no2i = 0.21 * afr * igd;
        nh2oi = (1.0-fgp) * fgfr * igd;
     /* Exit Flow Rate based on Oxygen, instead of Carbon */
        exito2 = (no2i + nh2oi) / (x[1] + x[2] + x[3] + x[4]);

        y[0]=x[4+N];
        y[1]=x[5+N];
        y[2]=exito2;
        y[3]=x[2];

}

/*
 * mdlUpdate - perform action at major integration time step
 *
 * This function is called once for every major integration time step.
 * Discrete states are typically updated here, but this function is useful
 * for performing any tasks that should only take place once per integration
 * step.
 */

static void mdlUpdate(double *x, double *u, SimStruct *S, int tid)
{
    /*
     * YOUR CODE GOES HERE
     */
}

/*
 * mdlDerivatives - compute the derivatives
 *
 * In this function, you compute the S-function block's derivatives.
 * The derivatives are placed in the dx variable.
 */

static void mdlDerivatives(double *dx, double *x, double *u, SimStruct *S, int tid)
{
    	int i,j;
	double igd, feed, vfl;
	double cch4i, ch2oi, co2i, cn2i;
	double nch4i, nh2oi, no2i, nn2i, ncoi, nco2i;
	double nch4f, nh2of, no2f, nn2f, ncof, nco2f;
	double qi, qf, qrxn1, qrxn2, qfluid;
	double sum,ntotf,exitt,exito2;
	double Ndiv,L,DL,D,H,PI,CSA,RHOFL,CPFL,TAUH;
	double EFFURN,CPFURN,VOL,K1,K2,HRXN1,HRXN2,P,R;
	double CPCH4I,CPH2OI,CPO2I,CPN2I,CPCOI,CPCO2I;
	double CPCH4F,CPH2OF,CPO2F,CPN2F,CPCOF,CPCO2F;
	double hcfr,hcit,afr,at,fgfr,fgt,fgp;

     /* Inputs */
	hcfr=u[0];	/* Hydrocarbon Flow Rate (m3/min) */
	hcit=u[1];	/* Hydrocarbon Inlet Temperature (K) */
	afr=u[2];	/* Air Flow Rate (m3/min) */
	at=u[3];	/* Air Temperature */
	fgfr=u[4];	/* Fuel Gas Flow Rate (m3/min) */
	fgt=u[5];	/* Fuel Gas Temperature (K) */
	fgp=u[6];	/* Fuel Gas Purity (mol CH4/mol) */


     /* furnace */
	EFFURN=.8712;        /* energy efficiency */
	CPFURN=1.05e+05;        /* heat capacity (J/K) */
	VOL=250.0;              /* reaction volume (m3) */

     /* fluid tubing */
	Ndiv=20.0;           /* Partition #, for division purposes */
	L=40.0;                 /* lenghth (m) */
	DL=(L/Ndiv);               /* length of partition */
	D=0.08;               /* diameter (m) */
	H=400.0;                /* heat transfer coeficient */
	PI=3.14159265358979;
	CSA=(PI*(D/2.0)*(D/2.0));   /* cross sectional area */

     /* fluid (Eicosane n-C2OH42 mp=?   bp=?  */
	RHOFL=786.6;          /* density (kg/m3) */
	CPFL=463.31;          /* heat capacity */
	TAUH=((RHOFL*CPFL*D)/(H*4));

     /* combustion reactions */
     /* CH4  +  3/2  O2  --> CO  +  2 H2O  : rxn 1 */
     /* CO  +  1/2 O2  -->  CO2  : rxn 2 */
	K1=400.0;               /* rxn 1 rate constant (m^4/mol^1.5-s) */
	K2=200.0;               /* rxn 2 rate constant (m^1.5/mol^1.5-s */
	HRXN1=6.0443e+05;       /* rxn 1 heat of reaction (J/mol) (T=1400K) */
	HRXN2=2.8498e+05;       /* rxn 2 heat of reaction (J/mol) (T=1400K) */

     /* initial heat capacities (J/mol-K) (T=310) */
	CPCH4I=35.826;        /* methane */
	CPH2OI=33.634;        /* water  */
	CPO2I=29.595;         /* oxygen */
	CPN2I=29.144;         /* nitrogen */
	CPCOI=29.235;         /* carbon monoxide */
	CPCO2I=38.053;        /* carbon dioxide */

     /* final heat capacities (J/mol-K) (T=1400K) */
	CPCH4F=84.586;        /* methane */
	CPH2OF=45.778;        /* water  */
	CPO2F=36.048;         /* oxygen */
	CPN2F=34.189;         /* nitrogen */
	CPCOF=34.538;         /* carbon monoxide */
	CPCO2F=57.042;        /* carbon dioxide */

     /* other constants */
	P=1.0;                  /* pressure */
	R=8.206e-05;           /* ideal gas constant (m^3-atm/mol-K) */

        igd=P/(R*at);       /* ideal gas density */
        feed=afr+fgfr;     /* combustion reactants feed rate */
        vfl=hcfr/CSA;       /* fluid velocity */

     /* initial concentrations */
        cch4i=fgp*fgfr*igd/feed;
        ch2oi=(1.0-fgp)*fgfr*igd/feed;
        co2i=.21*afr*igd/feed;
        cn2i=.79*afr*igd/feed;

     /* initial species flow rates */
        nch4i=cch4i*feed;
        nh2oi=ch2oi*feed;
        no2i=co2i*feed;
        nn2i=cn2i*feed;
        ncoi=0.0;
        nco2i=0.0;

    /* Final Species Flow Rates (m3/min) */
        nch4f=nch4i-K1*x[0]*(pow(x[2],1.5))*VOL; /* Methane */
        nh2of=nh2oi+2.0*K1*x[0]*(pow(x[2],1.5))*VOL; /* Water */
        no2f=no2i-1.5*K1*x[0]*(pow(x[2],1.5))*VOL-0.5*K2*x[3]*pow(x[2],0.5)*VOL; /* Oxygen */
        nn2f=nn2i; /*Nitrogen */
        ncof=K1*x[0]*pow(x[2],1.5)*VOL-K2*x[3]*pow(x[2],0.5)*VOL; /*Carbon Monoxide */
        nco2f=K2*x[3]*pow(x[2],0.5)*VOL; /* Carbon Dioxide */

        ntotf=nch4f+nh2of+no2f+nn2f+ncof+nco2f; /* Total Molar Flow */

    /* Combustion Products Exit Flow (m3/min), from C balance */

        exitt=nch4i/(x[0]+x[3]+x[4]);

    /* Combustion Products Exit Flow (m3/min), from O2 balance */

        exito2=(no2i + nh2oi) / (x[1] + x[2] + x[3] + x[4]);

        dx[0]=(1.0/VOL)*(nch4i-exitt*x[0])-K1*x[0]*(pow(x[2],1.5));
        dx[1]=(1.0/VOL)*(nh2oi-exitt*x[1])+2.0*K1*x[0]*(pow(x[2],1.5));
        dx[2]=(1.0/VOL)*(no2i-exitt*x[2])-1.5*K1*x[0]*(pow(x[2],1.5))-0.5*K2*x[3]*(pow(x[2],0.5));
        dx[3]=(1.0/VOL)*(ncoi-exitt*x[3])+K1*x[0]*(pow(x[2],1.5))-K2*x[3]*(pow(x[2],0.5));
        dx[4]=(1.0/VOL)*(nco2i-exitt*x[4])+K2*x[3]*(pow(x[2],0.5));
        dx[5]=(vfl/DL)*(hcit-x[5])+(1/TAUH)*(x[5+N]-x[5]);
   
        for (i=6;i<N+5;++i){
           dx[i]=(vfl/DL)*(x[i-1]-x[i])+(1/TAUH)*(x[5+N]-x[i]);
        }
    /* nrg consideration */
    /* heat exchanger */
        qi=(nch4i*CPCH4I+nh2oi*CPH2OI+no2i*CPO2I+nn2i*CPN2I)*at;
        qf=(nch4f*CPCH4F+nh2of*CPH2OF+no2f*CPO2F+nn2f*CPN2F+ncof*CPCOF+nco2f*CPCO2F)*x[5+N];
   
    /* generation */     
        qrxn1=EFFURN*(K1*x[0]*(pow(x[2],1.5))*VOL)*HRXN1;
        qrxn2=EFFURN*(K2*x[3]*(pow(x[2],0.5))*VOL)*HRXN2;

    /* nrg lost to fluid */
        sum=0.0;
        for (j=5;j<5+N;++j){
           sum=sum+x[j];
        }   
        qfluid=H*D*DL*PI*(Ndiv*x[5+N]-sum);
        dx[5+N]=(qrxn1+qrxn2-(qf-qi)-qfluid)/CPFURN;

}


#if 0  /* Change to a 1 if VARIABLE_SAMPLE_TIME has been specified */
/*
 * mdlGetTimeOfNextVarHit - Get the time of the next variable sample time hit
 *  
 * This function is called one for every major integration time step.
 * It must return time of next hit by using ssSetTNext. The time of the
 * next hit must be greater than ssGetT(S).
 */
#define MDL_GET_TIME_OF_NEXT_VAR_HIT
static void mdlGetTimeOfNextVarHit(SimStruct *S)
{
    /* ssSetTNext(S, <timeOfNextHit>); */
}
#endif

/*
 * mdlTerminate - called when the simulation is terminated.
 *
 * In this function, you should perform any actions that are necessary
 * at the termination of a simulation.  For example, if memory was allocated
 * in mdlInitializeConditions, this is the place to free it.
 */

static void mdlTerminate(SimStruct *S)
{
    /*
     * YOUR CODE GOES HERE
     */
}

#ifdef	MATLAB_MEX_FILE    /* Is this file being compiled as a MEX-file? */
#include "simulink.c"      /* MEX-file interface mechanism */
#else
#include "cg_sfun.h"       /* Code generation registration function */
#endif
